<?php

/**
 * @package Unlimited Elements
 * @author UniteCMS http://unitecms.net
 * @copyright Copyright (c) 2016 UniteCMS
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU/GPLv2 or later
 */

defined('UNLIMITED_ELEMENTS_INC') or die('Restricted access');

abstract class UCAdminNoticeAbstract{

	protected $condition; // no_doubly
	protected $start = 0; // offset in seconds from the plugin installation
	protected $duration = 0; // offset in seconds from the first display
	protected $freeOnly = false; // show the notice only in the free version
	protected $debug = false; // force show the notice with metadata
	
	
	/**
	 * get the notice identifier
	 */
	abstract public function getId();

	/**
	 * get the notice html
	 */
	abstract public function getHtml();

	/**
	 * init the notice
	 */
	abstract public function init();
	
	
	/**
	 * construct the notice
	 */
	public function __construct(){
		
		$this->init();
	}
	
	
	/**
	 * check if the notice should be displayed
	 */
	public function shouldDisplay(){
		
		$isDebug = $this->isDebug();
		
		if($isDebug === true){
			return true;
		}

		$isDismissed = $this->isDismissed();

		if($isDismissed === true)
			return false;

		$isFreeAllowed = $this->isFreeAllowed();

		if($isFreeAllowed === false)
			return false;

		$isConditionAllowed = $this->isConditionAllowed();

		if($isConditionAllowed === false)
			return false;

		$isScheduled = $this->isScheduled();

		if($isScheduled === true)
			return false;

		return true;
	}

	/**
	 * mark the notice as dismissed
	 */
	public function dismiss(){

		$this->setOption('dismissed', true);
	}

	/**
	 * postpone the notice for the given duration (in seconds)
	 */
	public function postpone($duration){

		$this->setOption('start_time', time() + $duration);
		$this->deleteOption('finish_time');
	}

	/**
	 * set the debug mode
	 */
	public function setDebugMode(){
		
		$this->debug = true;
	}
	
	
	/**
	 * create a new builder instance
	 */
	protected function createBuilder(){
		
		$debug = '';

		$isDebug = $this->isDebug();
		
		$noticeID = $this->getId();
		
		$builder = new UCAdminNoticeBuilder($noticeID);

		if($isDebug == true){
			$debug = $this->getDebugData();
			$builder->debug($debug);
		}
		
		return $builder;
	}
	
	
	/**
	 * check if the notice is in the debug mode
	 */
	private function isDebug(){

		return $this->debug;
	}

	/**
	 * get the debug data of the notice
	 */
	private function getDebugData(){

		$isDismissed = $this->isDismissed();

		if($isDismissed === true)
			return 'Notice hidden - dismissed';

		$isFreeAllowed = $this->isFreeAllowed();

		if($isFreeAllowed === false)
			return 'Notice hidden - free only';

		$isConditionAllowed = $this->isConditionAllowed();

		if($isConditionAllowed === false)
			return 'Notice hidden - false condition';

		$isSchedule = $this->isScheduled();

		if($isSchedule === true)
			return 'Notice hidden - scheduled';

		return 'Notice visible';
	}

	/**
	 * check if the notice is dismissed
	 */
	private function isDismissed(){

		$isDismissed = $this->getOption('dismissed', false);
		$isDismissed = UniteFunctionsUC::strToBool($isDismissed);

		return $isDismissed;
	}

	/**
	 * check if the notice is allowed for the free version
	 */
	private function isFreeAllowed(){

		if($this->freeOnly === true && GlobalsUC::$isProVersion === true)
			return false;

		return true;
	}

	/**
	 * check if the notice condition is allowed
	 */
	private function isConditionAllowed(){

		switch($this->condition){
			case 'no_doubly':
				if(defined('DOUBLY_INC'))
					return false;
			break;
		}

		return true;
	}

	/**
	 * check if the notice is scheduled
	 */
	private function isScheduled(){

		$currentTime = time();
		$installTime = intval(UCAdminNoticesOptions::getOption('install_time'));
		$startTime = intval($this->getOption('start_time'));

		if(empty($startTime)){
			$startTime = $installTime + $this->start;
		}

		if($currentTime > $startTime){
			$finishTime = $this->getOption('finish_time');

			if(empty($finishTime)){
				$finishTime = $currentTime + $this->duration;

				$this->setOption('finish_time', $finishTime);
			}

			if($currentTime < $finishTime){
				return false;
			}
		}

		return true;
	}

	/**
	 * get the notice option value
	 */
	private function getOption($key, $fallback = null){

		$value = UCAdminNoticesOptions::getNoticeOption($this->getId(), $key, $fallback);

		return $value;
	}

	/**
	 * set the notice option value
	 */
	private function setOption($key, $value){

		UCAdminNoticesOptions::setNoticeOption($this->getId(), $key, $value);
	}

	/**
	 * delete the notice option value
	 */
	private function deleteOption($key){

		UCAdminNoticesOptions::deleteNoticeOption($this->getId(), $key);
	}

}
